/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Workfile: cifXHWFunctionsSample.c $
   Last Modification:
    $Author: Robert $
    $Modtime: 28.04.09 14:49 $
    $Revision: 2805 $
   
   Targets:
     ANSI         : yes
 
   Description:
     Demo application of the CIFX Hardware functions
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     10        27.08.2014  RM       Added HWIF function support, checked with CIFX_TOOLKIT_HWIF
                                    option setting
     9         21.05.2013  RM       Missing to change OS_Memcopy to HWIF_WRITEN in IODemo
                                    - handle output
     8         28.01.2013  RM       Updated sources to be usable for OS and nonOS examples
                                    - Converted examples from cpp sources to c sources
                                    - Moved OS specific example functions 
                                      (e.g. cifXTkHWFunctions_GetDPMPointer()) to separate
                                      c-module (cifXHWFunctionsOS.c)

     7         06.09.2012  SD       - Replaced DPM access (OS_Memcpy()) by the corresponding
                                      hardware read/write macro (HWIF_WRITEN/HWIF_READN)
                                    - Removed OS dependant code.

     6         16.08.2012  RM       Added example for indication handling     
     
     5         03.05.2012  RM       Added a configuration sample for POWERLINK

     4         19.03.2012  RM       cifXTkHWFunctions_InitializeDataStructures extended
                                    by checking for DPM is available including a timeout handling
     
     3         05.03.2012  RM       Added a sleep of 100ms before starting with device handling.
                                    Necessary becaus firmware needs a momnt to set the actual COS state
          
     2         16.08.2011  RM       - Set Board_NO to 0 to use the first available board
                                    - main() function restructured
                                    - fixing parameter oder in IODemo,  DEV_WaitForBitState() for
                                    the input image 

     1         17.03.2011  SD       initial version
     
**************************************************************************************/

#include "OS_Includes.h"
#include "cifXHWFunctions.h"
#include "cifXEndianess.h"

/* Error & Type Def. */
#include "CIFXErrors.h"

#ifdef CIFX_TOOLKIT_HWIF
  #include "SerialDPMInterface.h"
#endif

/* include files for Warmstart configuration packets */
#include "ProfibusAps_Public.h"
#include "PNSIF_API.h"
#include "PNIO_spec.h"
#include "Ecs_Public.h"
#include "Pls_Public.h"

/* general device functions */
#include "DEV_InterfaceUser.h"
#include "DEV_FileUser.h"

/*#define FIELDBUS_INDICATION_HANDLING*/
#define FIELDBUS_INDICATION_HANDLING
#ifdef FIELDBUS_INDICATION_HANDLING
  #include "Fieldbus_Functions.h"                          
#endif

int  cifXTkHWFunctions_GetDPMPointer( uint8_t** ppbDPM, uint32_t* pulDPMSize);
void cifXTkHWFunctions_FreeDPMPointer( uint8_t* pbDPM);

/* ============================================================================= */
/* Global infomation and definitions                                             */
/* ============================================================================= */

/* global trace level referenced by the hardware function module (cifXHWFunctions.c) */
uint32_t                  g_ulTraceLevel        = 0;

/* My source ID (packet communication)         */
uint32_t                  ulSourceID            = 0x43214321;

/* Cards protocol and communication class      */
uint8_t                   bHandleIndications    = 0;
uint16_t                  usProtocollClass      = 0;
uint16_t                  usCommunicationClass  = 0;

/* com channel number where firmware is placed */                  
#define                   COM_CHANNEL           0     /* Use the first Channel on the choosen board */ 

/* cycles to run IO_DEMO()                     */
#define                   DEMO_CYCLES           10000         
/* size of IO transfer data (bytes)            */
#define                   SIZE_OF_IODATA        64 

/* device instance channel count               */
#define                   DEV_CHANNEL_COUNT     COM_CHANNEL + 1     
#define                   COM_CHANNEL_OFFS      2
  

/* Return values of main */
#define DEV_NO_ERROR           0
/* DPM identification error                             */
#define ERR_DEV_DPM_IDENT      1
/* DEV_Init() failed!                                   */
#define ERR_DEV_INIT           2
/* DEV_Ready() request system channel failed!           */
#define ERR_DEV_SYS_READY      3
/* DEV_Ready() request communication channel failed!    */
#define ERR_DEV_COM_READY      4
/* Sending Warmstart packet failed!                     */
#define ERR_DEV_WARMSTART      5
/* DEV_Communicating() request failed!                  */
#define ERR_DEV_COM            6
/* DEV_Demo() failed!                                   */
#define ERR_DEV_DEMO           7
/* DPM mapping failed!                                  */
#define ERR_DEV_MAP            8
/* No configuration found!                              */
#define ERR_DEV_CONFIG         9
/* DPM Layout unknown                                   */
#define ERR_DEV_DPM_LAYOUT     10
/* Firmware missing                                     */
#define ERR_DEV_FW_MISSING     11



/*****************************************************************************/
/*! User trace function
*   right while cifXTKitAddDevice is being processed
*   \param ptDevInstance  Device instance
*   \param ulTraceLevel   Trace level
*   \param szFormat       Format string                                      */
/*****************************************************************************/
void USER_Trace(PDEVICEINSTANCE ptDevInstance, uint32_t ulTraceLevel, const char* szFormat, ...)
{
  UNREFERENCED_PARAMETER(ulTraceLevel);
  UNREFERENCED_PARAMETER(ptDevInstance);
  UNREFERENCED_PARAMETER(szFormat);
}


/*****************************************************************************/
/*! Setup device specific configuration ETHERCAT
 * \param ptSendPacket              configuration packet
 * \return CIFX_NO_ERROR on success                                          */
/*****************************************************************************/
static int32_t SetupECSConfiguration( CIFX_PACKET* ptSendPacket)
{
  int32_t                           lRet = CIFX_NO_ERROR;
  /* This is a EtherCAT device */
  ECAT_DPM_SET_CONFIGURATION_REQ_T* ptEcsSetConfigReq;

  ptEcsSetConfigReq = (ECAT_DPM_SET_CONFIGURATION_REQ_T*) ptSendPacket;

  OS_Memset( (void*)ptEcsSetConfigReq, 0, sizeof (*ptEcsSetConfigReq) );

  /* header resources used to address a protocoll stack, define len, ... */
  ptEcsSetConfigReq->tHead.ulDest           = HOST_TO_LE32(RCX_PACKET_DEST_DEFAULT_CHANNEL);
  ptEcsSetConfigReq->tHead.ulSrc            = HOST_TO_LE32(ulSourceID);
  ptEcsSetConfigReq->tHead.ulLen            = HOST_TO_LE32(sizeof (ptEcsSetConfigReq->tData));
  ptEcsSetConfigReq->tHead.ulCmd            = HOST_TO_LE32(ECAT_DPM_SET_CONFIGURATION_REQ);

  /* stack specific information             */
  /* example packet for netX500 V0.0 device */
  ptEcsSetConfigReq->tData.ulSystemFlags            = HOST_TO_LE32(0);
  ptEcsSetConfigReq->tData.ulDeviceType             = HOST_TO_LE32(0);
  /* Hilscher Vendor-Id                     */
  ptEcsSetConfigReq->tData.ulVendorId               = HOST_TO_LE32(0x00000044);
  /* Product-Code of a netX500 V0.0 device  */
  ptEcsSetConfigReq->tData.ulProductCode            = HOST_TO_LE32(0x00000009);
  ptEcsSetConfigReq->tData.ulRevisionNumber         = HOST_TO_LE32(0x00000000);

  ptEcsSetConfigReq->tData.ulSerialNumber           = HOST_TO_LE32(0);
  ptEcsSetConfigReq->tData.ulProcessDataOutputSize  = HOST_TO_LE32(4);
  ptEcsSetConfigReq->tData.ulProcessDataInputSize   = HOST_TO_LE32(4);

  return lRet;
}

/*****************************************************************************/
/*! Setup device specific configuration PROFINET
 * \param ptSendPacket              configuration packet
 * \return CIFX_NO_ERROR on success                                          */
/*****************************************************************************/
static int32_t SetupPNSConfiguration( CIFX_PACKET* ptSendPacket)
{
  int32_t                                 lRet = CIFX_NO_ERROR;
  /* This is a PROFINET device */
  PNS_IF_SET_CONFIGURATION_REQUEST_REQ_T* ptProfinetDeviceSetConfigReq = NULL;
  PNS_IF_API_STRUCT_T*                    ptApiStruct = NULL;
  PNS_IF_SUBMODULE_STRUCT_T*              ptSubmodule = NULL;

  ptProfinetDeviceSetConfigReq = (PNS_IF_SET_CONFIGURATION_REQUEST_REQ_T*)ptSendPacket;

  OS_Memset( (void*)ptProfinetDeviceSetConfigReq, 0, sizeof(*ptProfinetDeviceSetConfigReq));

  /* header resources used to address a protocoll stack, define len, ... */
  ptProfinetDeviceSetConfigReq->tHead.ulDest = HOST_TO_LE32(RCX_PACKET_DEST_DEFAULT_CHANNEL);
  ptProfinetDeviceSetConfigReq->tHead.ulSrc  = HOST_TO_LE32(ulSourceID);
  ptProfinetDeviceSetConfigReq->tHead.ulLen  = HOST_TO_LE32(sizeof (ptProfinetDeviceSetConfigReq->tData) + sizeof(PNS_IF_API_STRUCT_T) + 6 * sizeof(PNS_IF_SUBMODULE_STRUCT_T));
  ptProfinetDeviceSetConfigReq->tHead.ulCmd  = HOST_TO_LE32(PNS_IF_SET_CONFIGURATION_REQ);

  /* stack specific information             */
  /* example packet for netX500 V0.0 device */
  ptProfinetDeviceSetConfigReq->tData.ulTotalConfigPckLen = ptProfinetDeviceSetConfigReq->tHead.ulLen;

  OS_Strncpy((char*)ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.abNameOfStation, "netx500repns", 12);
  OS_Strncpy((char*)ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.abDeviceType   , "TestType"   ,  8);
  OS_Strncpy((char*)ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.abTypeOfStation, "TestTypeStation", 15);

  ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.ulSystemFlags        = HOST_TO_LE32(PNS_IF_SYSTEM_START_AUTO_START | PNS_IF_SYSTEM_STACK_HANDLE_I_M_ENABLED | PNS_IF_SYSTEM_ARDY_WOUT_APPL_REG_ENABLED);
  ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.ulCompleteInputSize  = HOST_TO_LE32(32);
  ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.ulCompleteOutputSize = HOST_TO_LE32(32);
  /* Device-Id of a netX500 RE/PNS          */
  ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.ulDeviceId           = HOST_TO_LE32(0x0107);
  /* Hilscher Vendor-Id                     */
  ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.ulVendorId           = HOST_TO_LE32(0x011E);
  ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.usInstanceId         = HOST_TO_LE16(1);
  ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.usMaxDiagRecords     = HOST_TO_LE16(16);
  ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.usSwRevision1        = HOST_TO_LE16(4);
  ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.usSwRevision2        = HOST_TO_LE16(0);
  ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.usSwRevision3        = HOST_TO_LE16(0);
  ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.bSwRevisionPrefix    = 'T';

  ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.ulNameOfStationLen   = HOST_TO_LE32((TLR_UINT32)OS_Strlen((char*)ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.abNameOfStation));
  ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.ulTypeOfStationLen   = HOST_TO_LE32((TLR_UINT32)OS_Strlen((char*)ptProfinetDeviceSetConfigReq->tData.tDeviceParameters.abTypeOfStation));

  ptProfinetDeviceSetConfigReq->tData.tModuleConfig.ulNumApi                 = HOST_TO_LE32(1);

  ptApiStruct = (PNS_IF_API_STRUCT_T*)(&(ptProfinetDeviceSetConfigReq->tData.tModuleConfig) + 1);

  ptApiStruct->ulApi                                                         = HOST_TO_LE32(0);
  ptApiStruct->ulNumSubmodItems                                              = HOST_TO_LE32(6);

  ptSubmodule = (PNS_IF_SUBMODULE_STRUCT_T*)(ptApiStruct + 1);

  memset(ptSubmodule,0x00,sizeof(*ptSubmodule));
  ptSubmodule->usSlot               = HOST_TO_LE16(0);
  ptSubmodule->usSubslot            = HOST_TO_LE16(1);
  ptSubmodule->ulModuleId           = HOST_TO_LE32(0x00001001);
  ptSubmodule->ulSubmodId           = HOST_TO_LE32(0x00001000);

  ptSubmodule++;

  memset(ptSubmodule,0x00,sizeof(*ptSubmodule));
  ptSubmodule->usSlot               = HOST_TO_LE16(0);
  ptSubmodule->usSubslot            = HOST_TO_LE16(0x8000);
  ptSubmodule->ulModuleId           = HOST_TO_LE32(0x00001001);
  ptSubmodule->ulSubmodId           = HOST_TO_LE32(0x00001001);

  ptSubmodule++;

  memset(ptSubmodule,0x00,sizeof(*ptSubmodule));
  ptSubmodule->usSlot               = HOST_TO_LE16(0);
  ptSubmodule->usSubslot            = HOST_TO_LE16(0x8001);
  ptSubmodule->ulModuleId           = HOST_TO_LE32(0x00001001);
  ptSubmodule->ulSubmodId           = HOST_TO_LE32(0x00001001);

  ptSubmodule++;

  memset(ptSubmodule,0x00,sizeof(*ptSubmodule));
  ptSubmodule->usSlot               = HOST_TO_LE16(0);
  ptSubmodule->usSubslot            = HOST_TO_LE16(0x8002);
  ptSubmodule->ulModuleId           = HOST_TO_LE32(0x00001001);
  ptSubmodule->ulSubmodId           = HOST_TO_LE32(0x00001001);

  ptSubmodule++;

  memset(ptSubmodule,0x00,sizeof(*ptSubmodule));
  ptSubmodule->usSlot               = HOST_TO_LE16(1);
  ptSubmodule->usSubslot            = HOST_TO_LE16(1);
  ptSubmodule->ulModuleId           = HOST_TO_LE32(0x00000012);
  ptSubmodule->ulSubmodId           = HOST_TO_LE32(0x00000011);
  ptSubmodule->ulProvDataLen        = HOST_TO_LE32(32);
  ptSubmodule->ulConsDataLen        = HOST_TO_LE32(0);

  ptSubmodule++;

  memset(ptSubmodule,0x00,sizeof(*ptSubmodule));
  ptSubmodule->usSlot               = HOST_TO_LE16(2);
  ptSubmodule->usSubslot            = HOST_TO_LE16(1);
  ptSubmodule->ulModuleId           = HOST_TO_LE32(0x00000013);
  ptSubmodule->ulSubmodId           = HOST_TO_LE32(0x00000012);
  ptSubmodule->ulProvDataLen        = HOST_TO_LE32(0);
  ptSubmodule->ulConsDataLen        = HOST_TO_LE32(32);

  return lRet;
}

/*****************************************************************************/
/*! Setup device specific configuration POWERLINK
 * \param ptSendPacket              configuration packet
 * \return CIFX_NO_ERROR on success                                          */
/*****************************************************************************/
static int32_t SetupPLSConfiguration( CIFX_PACKET* ptSendPacket)
{
  int32_t                   lRet = CIFX_NO_ERROR;
  /* This is a POWERLINK device */
  EPLCN_DPM_WARMSTART_REQ_T *ptWarmstartReq;
 
  /* initialize the Warmstart request packet */
  ptWarmstartReq = (EPLCN_DPM_WARMSTART_REQ_T *)ptSendPacket;

  OS_Memset( (void*)ptWarmstartReq, 0, sizeof(*ptWarmstartReq));

  /* header resources used to address a protocoll stack, define len, ... */
  ptWarmstartReq->tHead.ulDest                    = HOST_TO_LE32(RCX_PACKET_DEST_DEFAULT_CHANNEL); /* Destination of packet                 */
  ptWarmstartReq->tHead.ulSrc                     = HOST_TO_LE32(ulSourceID);                   /* Source of packet, process queue          */
  ptWarmstartReq->tHead.ulLen                     = HOST_TO_LE32(sizeof(ptWarmstartReq->tData));/* Length of packet data without header     */
  ptWarmstartReq->tHead.ulCmd                     = HOST_TO_LE32(EPLCN_DPM_WARMSTART_REQ);      /* Packet command                           */

  /* stack specific information            */
  /* example packet for netX PLS device */
  ptWarmstartReq->tData.ulSystemFlags             = HOST_TO_LE32(0);                            /* System Flags                             */
  ptWarmstartReq->tData.ulWatchdogTime            = HOST_TO_LE32(1000);                         /* Watchdog time                            */
  ptWarmstartReq->tData.ulVendorId                = HOST_TO_LE32(0x44);                         /* Vendor Id                                */

  /* TODO: modify*/
  ptWarmstartReq->tData.ulProductCode             = HOST_TO_LE32(0x00000019);                   /* Product code of CifX example application */
  ptWarmstartReq->tData.ulRevisionNumber          = HOST_TO_LE32(0);                            /* Revision number                          */
  ptWarmstartReq->tData.ulSerialNumber            = HOST_TO_LE32(0);                            /* Serial number                            */
  ptWarmstartReq->tData.ulStackConfigurationFlags = HOST_TO_LE32(MSK_EPLCN_DPM_WARMSTART_STACK_CFG_CONFIGURE_DEFAULT_OBJECTS |
                                                                 MSK_EPLCN_DPM_WARMSTART_STACK_CFG_DISABLE_PDO_MAP_VERS_CHECK);
                                                                                  /* Configuration Flags                      */
  ptWarmstartReq->tData.ulThresholdDisableFlags   = HOST_TO_LE32(0);                            /* Threshold Disable Flags                  */
  ptWarmstartReq->tData.ulThresholdLossSoC        = HOST_TO_LE32(0);                            /* LossSoC Threshold   (ignored if set to 0)*/
  ptWarmstartReq->tData.ulThresholdLossPReq       = HOST_TO_LE32(0);                            /* LossPReq Threshold  (ignored if set to 0)*/
  ptWarmstartReq->tData.ulThresholdLossSoA        = HOST_TO_LE32(0);                            /* LossSoA Threshold   (ignored if set to 0)*/
  ptWarmstartReq->tData.ulThresholdSoCJitter      = HOST_TO_LE32(0);                            /* SoCJitter Threshold (ignored if set to 0)*/
  ptWarmstartReq->tData.ulThresholdCollision      = HOST_TO_LE32(0);                            /* Collision Threshold (ignored if set to 0)*/
  ptWarmstartReq->tData.ulThresholdCrcError       = HOST_TO_LE32(0);                            /* CrcError Threshold  (ignored if set to 0)*/
  ptWarmstartReq->tData.ulCycleLength             = HOST_TO_LE32(1000);                         /* Cycle Length configuration (ns)          */
  ptWarmstartReq->tData.ulSoCJitterRange          = HOST_TO_LE32(2000);                         /* SoC Jitter range configuration (ns)      */
  ptWarmstartReq->tData.usProcessDataOutputSize   = HOST_TO_LE16(32);                           /* Process Data Output Size in Bytes        */
  ptWarmstartReq->tData.usProcessDataInputSize    = HOST_TO_LE16(32);                           /* Process Data Input Size in Bytes         */
  memcpy(ptWarmstartReq->tData.abNodeName, "netX500 100 EXAMPLE PLS", sizeof(ptWarmstartReq->tData.abNodeName));
                                                                                  /* EPL node name                            */
  ptWarmstartReq->tData.ulGatewayAddress          = HOST_TO_LE32(0xC0A864FE);     /* IP Gateway: 192.168.100.254              */

  ptWarmstartReq->tData.bNodeId                   = 1;                            /* EPL node id (range 1 to 239)             */

  ptWarmstartReq->tData.bSoCTriggerConfig         = 0;                            /* SoC Trigger configuration                */
  ptWarmstartReq->tData.ulSoCTriggerDelay         = HOST_TO_LE32(0);              /* SoC Trigger Delay configuration          */
  ptWarmstartReq->tData.ulSoCTriggerLength        = HOST_TO_LE32(0);              /* SoC Trigger Length configuration         */
  ptWarmstartReq->tData.bPReqMappingVersion       = 0;                            /* PReq Mapping Version                     */
  ptWarmstartReq->tData.bPResMappingVersion       = 0;                            /* PRes Mapping Version                     */
  ptWarmstartReq->tData.bNumberOfStatusEntries    = 0;                            /* configured status entries                */

  return lRet;
}

/*****************************************************************************/
/*! Setup device specific configuration PROFIBUS
 * \param ptSendPacket              configuration packet
 * \return CIFX_NO_ERROR on success                                          */
/*****************************************************************************/
static int32_t SetupPBSConfiguration( CIFX_PACKET* ptSendPacket)
{
  int32_t                                       lRet = CIFX_NO_ERROR;
  /* This is a PROFIBUS device */
  PROFIBUS_APS_PACKET_SET_CONFIGURATION_REQ_T*  ptProfibusSetConfigReq; 

  /* initialize the Set Configuration packet with two specific bytes describing the I/O capabilities of the example slave device */
  ptProfibusSetConfigReq = (PROFIBUS_APS_PACKET_SET_CONFIGURATION_REQ_T*)ptSendPacket;

  OS_Memset( (void*)ptProfibusSetConfigReq, 0, sizeof (*ptProfibusSetConfigReq) );

  /* header resources used to address a protocoll stack, define len, ... */
  ptProfibusSetConfigReq->tHead.ulDest        = HOST_TO_LE32(RCX_PACKET_DEST_DEFAULT_CHANNEL);
  ptProfibusSetConfigReq->tHead.ulSrc         = HOST_TO_LE32(ulSourceID);
  ptProfibusSetConfigReq->tHead.ulLen         = HOST_TO_LE32(sizeof (ptProfibusSetConfigReq->tData) - sizeof (ptProfibusSetConfigReq->tData.abCfgData) + 2);
  ptProfibusSetConfigReq->tHead.ulCmd         = HOST_TO_LE32(PROFIBUS_APS_SET_CONFIGURATION_REQ);

  /* stack specific information            */
  /* example packet for netX DP/DPS device */
  ptProfibusSetConfigReq->tData.ulSystemFlags = HOST_TO_LE32(0);
  ptProfibusSetConfigReq->tData.ulWdgTime     = HOST_TO_LE32(0);
  /* ident number of netX DP/DPS           */
  ptProfibusSetConfigReq->tData.usIdentNumber = HOST_TO_LE32(0x0A12);
  ptProfibusSetConfigReq->tData.bBusAddr      = 3;
  /* automatic detection                   */
  ptProfibusSetConfigReq->tData.bBaudRate     = 15;     
  ptProfibusSetConfigReq->tData.bFlags        = 0x07;
  ptProfibusSetConfigReq->tData.bCfgLen       = 2;
  ptProfibusSetConfigReq->tData.abCfgData[0]  = 0x91;
  ptProfibusSetConfigReq->tData.abCfgData[1]  = 0xA1;

  return lRet;
}

/*****************************************************************************/
/*! Identify protocoll and create configuration packet
 * TODO: Build configuration packet
 * IdentifyWarmstartPacket() is an example for PROFIBUS / PROFINET / ETHERCAT Slave
 *
 * \param ptSendPacket              configuration packet
 * \return CIFX_NO_ERROR on success                                          */
/*****************************************************************************/
int32_t ProcessDeviceWarmstart(  CHANNELINSTANCE* ptChannel)
{
  int32_t         lRet                  = CIFX_NO_ERROR;

  CIFX_PACKET     tSendPacket           = {0};    /* Create a default rcX packet buffer */
  CIFX_PACKET*    ptSendPacket          = &tSendPacket;

  /* Set pointer to the device instance structure and the system channel */
  PDEVICEINSTANCE      ptDevInst        = (PDEVICEINSTANCE)ptChannel->pvDeviceInstance;
  NETX_SYSTEM_CHANNEL* ptSystemChannel  = (NETX_SYSTEM_CHANNEL*)ptDevInst->tSystemDevice.pbDPMChannelStart;

  /* retrieve "ProtocollClass" and "CommunicationClass" from the DPM */
  usProtocollClass     = LE16_TO_HOST(HWIF_READ16(ptDevInst, ptSystemChannel->atChannelInfo[ptChannel->ulChannelNumber + COM_CHANNEL_OFFS].tCom.usProtocolClass));
  usCommunicationClass = LE16_TO_HOST(HWIF_READ16(ptDevInst, ptSystemChannel->atChannelInfo[ptChannel->ulChannelNumber + COM_CHANNEL_OFFS].tCom.usCommunicationClass));

  /* retrieve the protocoll to fill the "Warmstart" packet with the corresponding parameter */
  switch(usProtocollClass)
  { 
    /* PROFINET */
    case RCX_PROT_CLASS_PROFINET_IO:
    {
      /* check if it is a slave */
      if (RCX_COMM_CLASS_IO_DEVICE != usCommunicationClass)
      { 
        /* Error! Device is not a Slave */ 
        lRet = CIFX_INVALID_PARAMETER;
      } else
      {
        /* PROFINET Device found! Setup configuration for netX500 RE/PNS. */
        lRet = SetupPNSConfiguration(ptSendPacket);
      }
      break;
    }
    /* PROFIBUS */
    case RCX_PROT_CLASS_PROFIBUS_DP:
    {
      /* check if it is a slave */
      if (RCX_COMM_CLASS_SLAVE != usCommunicationClass)
      {
        /* Error! Device is not a Slave */ 
        lRet = CIFX_INVALID_PARAMETER;

      }  else
      { 
        /* PROFIBUS Slave found! Setup configuration for  netX DP/DPS. */
        lRet = SetupPBSConfiguration(ptSendPacket);
      }
      break;
    }
    /* ETHERCAT */
    case RCX_PROT_CLASS_ETHERCAT:
    {
      /* check if it is a slave */
      if (RCX_COMM_CLASS_SLAVE != usCommunicationClass)
      {
        /* Error! Device is not a Slave */
        lRet = CIFX_INVALID_PARAMETER;

      } else
      {
        /* ETHERCAT Slave found! Setup configuration for netX500 V0.0. */
        lRet = SetupECSConfiguration(ptSendPacket);
      }
      break;
    }
    /* POWERLINK */
    case RCX_PROT_CLASS_POWERLINK:
    {
      /* check if it is a slave */
      if (RCX_COMM_CLASS_CONTROLLED_NODE != usCommunicationClass)
      {
        /* Error! Device is not a Slave */
        lRet = CIFX_INVALID_PARAMETER;

      } else
      {
        /* ETHERCAT Slave found! Setup configuration for netX500 V0.0. */
        lRet = SetupPLSConfiguration(ptSendPacket);
      }
      break;
    }
    default:
    {
      /* Unknown device */
      lRet = CIFX_INVALID_PARAMETER;
    
      break;
    }
  }

  if( CIFX_NO_ERROR == lRet)
  {
    /*------------------------------------------------------------------------*/
    /* Process "Warmstart" will configure the communication channel...        */
    /*------------------------------------------------------------------------*/
    lRet = DEV_ProcessWarmstart( ptChannel, &tSendPacket);
  }

  return lRet;
}

/*****************************************************************************/
/*! I/O Demo (read inputs and write back to outputs)
/ * \param ptDevInstance Device instance
/ * \return CIFX_NO_ERROR on success                                          */
/*****************************************************************************/
int32_t IODemo(PCHANNELINSTANCE ptChannel)
{
  uint8_t     abData[SIZE_OF_IODATA]  = {0};
  int32_t     lRet                    = CIFX_NO_ERROR;
  uint8_t     fHandshakeMode          = 0;
  PIOINSTANCE ptCommIn                = NULL;
  PIOINSTANCE ptCommOut               = NULL;

  /* Execute a I/O data transfer */
  /* Initialize pointer to the Input and Output process data image from the hardware */
  ptCommIn  = ptChannel->pptIOInputAreas[0];
  ptCommOut = ptChannel->pptIOOutputAreas[0];

  /*------------------------------*/
  /* Handle Input data area       */
  /*------------------------------*/
  /* Read the actual II/O data handshake mode for the input areas */
  fHandshakeMode = DEV_GetIOBitstate(ptChannel, ptCommIn, 0);

  /* Check handshake mode mode */
  if(RCX_FLAGS_NONE == fHandshakeMode)
  {
    
    /* This is RCX_IO_MODE_UNCONTROLLED, copy data without using the handshake flags */
    
    HWIF_READN( ptChannel->pvDeviceInstance, abData, ptCommIn->pbDPMAreaStart, sizeof(abData));

  } else
  {
    
    /* This is a "controlled" mode, synchronized via the handshake flags */
    /* Check if the handshake flags are in the correct state, otherwise wait */
    /* until the state is reached and it is allowed to access the INPUT process data image */

    if(!DEV_WaitForBitState( ptChannel, ptCommIn->bHandshakeBit, fHandshakeMode, 100))
    {
      /* Failed to get the correct handshake flag state */
      lRet = CIFX_DEV_EXCHANGE_FAILED;
    } else
    {
      /* Read access is allowed, copy data to user buffer */
      HWIF_READN( ptChannel->pvDeviceInstance, abData, ptCommIn->pbDPMAreaStart, sizeof(abData));

      /* Signal / Achnowledge -> "Read Data Done" */
      DEV_ToggleBit( ptChannel, 1UL << ptCommIn->bHandshakeBit); 
    }
  }

  /*------------------------------*/
  /* Handle Output data area      */
  /*------------------------------*/

  fHandshakeMode = DEV_GetIOBitstate(ptChannel, ptCommOut, 1);

  /* Check transfer mode */
  if(RCX_FLAGS_NONE == fHandshakeMode)
  {
    
    /* This is RCX_IO_MODE_UNCONTROLLED, copy data without using the handshake flags */
    HWIF_WRITEN( ptChannel->pvDeviceInstance, ptCommOut->pbDPMAreaStart, (void*)abData, sizeof(abData));
    
  } else
  {

    /* This is a "controlled" mode, synchronized via the handshake flags */
    /* Check if the handshake flags are in the correct state, otherwise wait */
    /* until the state is reached and it is allowed to access the OUTPUT process data image */

    if(!DEV_WaitForBitState( ptChannel, ptCommOut->bHandshakeBit, fHandshakeMode, 100))
    {
      lRet = CIFX_DEV_EXCHANGE_FAILED;
    } else
    {
      /* Write data */
      HWIF_WRITEN( ptChannel->pvDeviceInstance, ptCommOut->pbDPMAreaStart, (void*)abData, sizeof(abData));
      
      /* Signal -> "Write Data Done" */
      DEV_ToggleBit( ptChannel, 1UL << ptCommOut->bHandshakeBit); 
    }
  }

  /* Return information about communication state */
  DEV_IsCommunicating( ptChannel, &lRet);

  return lRet;
}


/*****************************************************************************/
/*! Packet Demo (read / write acyclic packets
/ * \param ptDevInstance Device instance
/ * \return CIFX_NO_ERROR on success                                          */
/*****************************************************************************/
void PacketDemo(PCHANNELINSTANCE ptChannel)
{
  int32_t   lRet            = CIFX_NO_ERROR;
  uint32_t  ulReceivePktCnt = 0;
  uint32_t  ulSendPktCnt    = 0;

  CIFX_PACKET tRecvPkt = {0};

  /*------------------------------*/
  /* Read the mailboxe state      */
  /*------------------------------*/
  lRet = DEV_GetMBXState( ptChannel, &ulReceivePktCnt, &ulSendPktCnt);
  if (CIFX_NO_ERROR != lRet)
    /* Unable to query the packet mailbox states */
    return;

  /*------------------------------*/
  /* Process receive packets      */
  /*------------------------------*/
  while  (ulReceivePktCnt > 0)
  {
    /* Try to read a packet from the mailbox */
    lRet = DEV_GetPacket( ptChannel, &tRecvPkt, sizeof(tRecvPkt), 0);
    
    if( CIFX_NO_ERROR != lRet)
    {
      /* Error reading a packet */

      break;
    }
    
    /* Check if we have acommand or answer packet */
    if( 0 == (LE32_TO_HOST(tRecvPkt.tHeader.ulCmd) & 1))
    { 
      /*-------------------------------------*/
      /* this is a commad / indication packet*/
      /*-------------------------------------*/

      /* TODO: handle the indication packet */

    } else
    {
      /*-------------------------------------*/
      /* this is a answer / response packet  */
      /*-------------------------------------*/
      /* we have a packet, check for packet error */
      if( 0 == LE32_TO_HOST(tRecvPkt.tHeader.ulState))
      {
        /* Answer with error state */

        /* TODO: handle respons packet */

      } else
      {
        /* Answer without error state */
        
        /* TODO: handle respons packet */

      }
    }
    ulReceivePktCnt--;

  } /* end while */

  /*------------------------------*/
  /* Process send packets         */
  /*------------------------------*/

  if (ulSendPktCnt > 0)
  {
    /* We can send a packt, and the card is able to accept it */

    /* TODO: handle your request packets */

  }
}


/*****************************************************************************/
/*! Initialize the necessary data structures
* \param pbDPM           Pointer to the start of the netX Dual Port Memory
* \param ulDPMSize       Size of the netX Memory in bytes
* \param ptDevInstance   Pointer toa device instance structure
* \param ulStartTimeout  Wait time until hardware should be up and running
* \return DEV_NO_ERROR on success                                            */
/*****************************************************************************/
unsigned long cifXTkHWFunctions_InitializeDataStructures( uint8_t* pbDPM, uint32_t ulDPMSize, PDEVICEINSTANCE ptDevInstance, unsigned long ulStartTimeout)
{
  /* Check if firmware is READY because we need the DPM Layout */
  typedef struct SYSTEM_STARTUP_LAYOUTtag
  {
    NETX_SYSTEM_CHANNEL  tSystemChannel;
    NETX_HANDSHAKE_ARRAY tHskChannel;
  } SYSTEM_STARTUP_LAYOUT;

  SYSTEM_STARTUP_LAYOUT*  ptSysStartup = NULL;
  unsigned long           lRet         = DEV_NO_ERROR;
  uint32_t ulDiffTime   = 0L;
  int32_t  lStartTime   = 0L;
  
  /* Cleanup and initialize the DEVICEINSTANCE structure */
  OS_Memset(ptDevInstance, 0, sizeof(*ptDevInstance));
  ptDevInstance->pbDPM     = pbDPM;
  ptDevInstance->ulDPMSize = ulDPMSize;

  /* Set pointer to handshake channel*/
  ptSysStartup = (SYSTEM_STARTUP_LAYOUT*)pbDPM;

  /*=====================================================================================================*/
  /* Now we have to wait until the System-Channel is ready, because we need a running system channel for */
  /* the further card handling                                                                           */
  /*=====================================================================================================*/
  /* Wait until System-Channel is READY */
  lStartTime   = (int32_t)OS_GetMilliSecCounter();

  /* This is the system channel which will reset the whole card */
  do
  {
    uint32_t ulHskFlags;
    ulHskFlags = LE32_TO_HOST(HWIF_READ32( ptDevInstance, ptSysStartup->tHskChannel.atHsk[0].ulValue));
    
    /* Check if DPM content is valid */
    if( (ulHskFlags == CIFX_DPM_INVALID_CONTENT)     ||
        (ulHskFlags == CIFX_DPM_NO_MEMORY_ASSIGNED)  ||
        (0 == (HWIF_READ8( ptDevInstance, ptSysStartup->tHskChannel.atHsk[0].t8Bit.bNetxFlags) & NSF_READY)))
    {
      /* Card is NOT READY */
      lRet = ERR_DEV_SYS_READY;
    } else
    {
      /* Card is READY */
      lRet = DEV_NO_ERROR;
      break;
    }
    /* Check time */
    ulDiffTime = OS_GetMilliSecCounter() - lStartTime;

    OS_Sleep(1);

  } while (ulDiffTime < ulStartTimeout);

  if( DEV_NO_ERROR == lRet)
  {
    /*===========================================================================*/ 
    /* Check the DPM "cookie" and find out if a firmware is running on the board */
    /* At least "BOOT" or "netX" must be available in the cookie                 */
    /*===========================================================================*/
    uint8_t abCookie[4];
    HWIF_READN( ptDevInstance, (void*)abCookie, pbDPM, sizeof(abCookie));
    
    if( 0 == OS_Memcmp( abCookie, &CIFX_DPMSIGNATURE_BSL_STR, sizeof(abCookie)))
    {
      /* Just the 2nd Stage Bootloader ist active !!! */
      /* Card needs a Firmware for correct funtionality */
      lRet = ERR_DEV_FW_MISSING;

    } else if( 0 == OS_Memcmp( abCookie, &CIFX_DPMSIGNATURE_FW_STR, sizeof(abCookie)))
    {

      /*===========================================================================*/ 
      /* Initialize all necessary data structures for the DEV function handling    */
      /*===========================================================================*/ 
      if (CIFX_NO_ERROR != ((lRet = DEV_Initialize(ptDevInstance, DEV_CHANNEL_COUNT))))
      {
        /* Device initialization failed! */
        lRet = ERR_DEV_INIT;
      }

    } else
    {
      /* DPM idendification failed! No valid Cookie! cifX/comX card is not present */
      lRet = ERR_DEV_DPM_IDENT;
    }
  }
  return lRet;
}


/*****************************************************************************/
/*! Uninitialize the data structures
*                                                                            */
/*****************************************************************************/
void cifXTkHWFunctions_UninitializeDataStructures( PDEVICEINSTANCE ptDevInstance)
{
  /* Uinitialize all data structures for the DEV function handling */
  DEV_Uninitialize( ptDevInstance);
}

/*****************************************************************************/
/*! Configure the selected device / communication channel
*
*   \return true on success                                                  */
/*****************************************************************************/
long cifXTkHWFunctions_ConfigureDevice( PCHANNELINSTANCE ptChannel, PCHANNELINSTANCE ptSysdevice)
{
  long lRet = DEV_NO_ERROR;

  /*----------------------------------------------------------------*/
  /* System device an communication channel must be ready           */
  /* Now check if the communication channel must be configured or   */
  /* if it us already "RUNNING"                                     */ 
  /*----------------------------------------------------------------*/

  /* if the running flag is set the channel is already configured,*/
  if ( DEV_IsRunning(ptChannel))
  { 

    /* TODO: Decide if the communication channel should be configured   */
    /*       on each start                                              */
    /*       If the device should be reconfigured, an already           */
    /*       existing configuration must be deleted first!!!            */

    /*------------------------------------------------------------------------*/
    /* Communication channel is RUNNING, which means it is configured         */
    /* To re-configure the channel, the existing configuration must be        */
    /* deleted first                                                          */
    /*------------------------------------------------------------------------*/
    DEV_RemoveChannelFiles(ptSysdevice, ptChannel->ulChannelNumber, DEV_TransferPacket, NULL, NULL, ".NXF");


    lRet = DEV_DeleteConfig( ptChannel);
    if( DEV_NO_ERROR != lRet) 
      lRet = ERR_DEV_CONFIG;

    /* Configure the communication channel again */
    lRet = ProcessDeviceWarmstart( ptChannel);
    if( CIFX_NO_ERROR != lRet)
      lRet = ERR_DEV_CONFIG;

  }else
  {
    /*------------------------------------------------------------------------*/
    /* Communication channel is not RUNNING, which means it is not configured */
    /* Try to configure the channel by using "Warmstart" parameters.          */
    /*                                                                        */
    /*------------------------------------------------------------------------*/
    lRet = ProcessDeviceWarmstart( ptChannel);
    if( CIFX_NO_ERROR != lRet)
      lRet = ERR_DEV_CONFIG;
  }

  return lRet;
}

/*****************************************************************************/
/*! Wait until communicating
*
*   \return true on success                                                  */
/*****************************************************************************/
long cifXTkHWFunctions_WaitUntilCommunicating( PCHANNELINSTANCE ptChannel)
{
  unsigned long ulConfigCnt = 100; 
  int32_t       lError      = 0; 
  int32_t       lRet        = DEV_NO_ERROR;

#ifdef FIELDBUS_INDICATION_HANDLING
  lError = Fieldbus_ActivateIndications( ptChannel, usProtocollClass, usCommunicationClass, &bHandleIndications);
#endif

  if( CIFX_NO_ERROR == lError)
  {
    /*------------------------------------------------------------------------*/
    /* After configuration, the channel is initialized                        */
    /* Usuallaly "start of communication" takes a few seconds,                */
    /* so we will wait 10 seconds until communication is running              */
    /*------------------------------------------------------------------------*/
    while ( ulConfigCnt--)
    { 
      if ( DEV_IsCommunicating( ptChannel, &lError))
      {
        /* DEV_IsCommunicating() suceeded communication should be available */
        break;
      }

      OS_Sleep(100);              /* Wait a bit */
    }
  }

  if (CIFX_NO_ERROR != lError)
    /* We do not reach the communication state */
    lRet = ERR_DEV_COM;

  return lRet;
}


/*****************************************************************************/
/*! Hardware function example
 * \return 0 on success                                                      */
/*****************************************************************************/
int32_t cifXHWSample( void)
{
  int32_t  lDemoRet  = DEV_NO_ERROR;
  int32_t  lRet      = CIFX_NO_ERROR;

  uint8_t*          pbDPM         = NULL;       /* This pointer must be loaded to the DPM address */
  uint32_t          ulDPMSize     = 0;          /* Size of the DPM in bytes */
  DEVICEINSTANCE    tDevInstance;               /* Global deveice data structure used by all DEV_xxx functions */

  /* Get pointer to the hardware dual-port memory and check if it is available */
  if ( FALSE == cifXTkHWFunctions_GetDPMPointer( &pbDPM, &ulDPMSize))
    /* Failed to get the hardware DPM pointer and size */
    return -1;

#ifdef CIFX_TOOLKIT_HWIF
  /* realizes serialDPM read/write access to the netX DPM interface */
  if ( SERDPM_UNKNOWN == SerialDPM_Init( &tDevInstance))
  {
    /* failed to initialize serial DPM functions */
  }
#endif 

  /* Initialize the necessary data structures */
  if ( DEV_NO_ERROR == cifXTkHWFunctions_InitializeDataStructures( pbDPM, ulDPMSize, &tDevInstance, 10000))
  {
    /*----------------------------------------------------------------*/
    /* Read actual device states                                      */
    /*----------------------------------------------------------------*/
    PCHANNELINSTANCE ptSystemDevice = &tDevInstance.tSystemDevice;
    PCHANNELINSTANCE ptChannel      = tDevInstance.pptCommChannels[COM_CHANNEL];

    /* Wait for State acknowlede by the firmware */
    OS_Sleep(100);              /* Wait a bit */

    /* read the host flags of the system device, first time to synchronize our internal status */
    DEV_ReadHostFlags( ptSystemDevice, 0);

    /* read the host flags of the communication channel, first time to synchronise our internal status */
    DEV_ReadHostFlags( ptChannel, 0);

    /* check if "system device" is ready... */
    if (!DEV_IsReady( ptSystemDevice))
    {
      /* System device is not ready! */
      lDemoRet = ERR_DEV_SYS_READY;

    /* check if "communication channel" is ready... */
    } else if ( !DEV_IsReady(ptChannel))
    {
      /* Communication channel is not ready! */
      lDemoRet = ERR_DEV_COM_READY;
      
    } else
    {
      /*------------------------------------------------------------------------*/
      /* At this point we should have a running device and a configured         */
      /* communication channel.                                                 */
      /* Procced with "NORMAL Stack Handling!                                   */
      /*------------------------------------------------------------------------*/
      /* Signal Host application is available */
      lRet = DEV_SetHostState( ptChannel, CIFX_HOST_STATE_READY, 1000);

      /* Configure the device */
      lDemoRet = cifXTkHWFunctions_ConfigureDevice( ptChannel, ptSystemDevice);
      //if( DEV_NO_ERROR != lDemoRet) 
      //  printf("Error");

      /* Initialize and activate interrupt if configured */
      DEV_InitializeInterrupt ( &tDevInstance);
      
      if (DEV_NO_ERROR == lDemoRet)
      {
        /*------------------------------------------------------------------------*/
        /* At this point we should have a running device and a configured         */
        /* communication channel if no error is shown                             */
        /*------------------------------------------------------------------------*/
        uint32_t ulState = 0;

        /* Signal Host application is available */
        lRet = DEV_SetHostState( ptChannel, CIFX_HOST_STATE_READY, 1000);

        /* Switch ON the BUS communication */
        lRet = DEV_BusState( ptChannel, CIFX_BUS_STATE_ON, &ulState, 3000);

        /* TODO: Decide to wait until communication is available or just go to */
        /*       to the cyclic data handling and check the state there         */
        /* Wait for communication is available or do this during the cyclic program handling*/
        lDemoRet = cifXTkHWFunctions_WaitUntilCommunicating( ptChannel);

        /*------------------------------------------------------------------------*/
        if (lDemoRet == DEV_NO_ERROR)
        {
          /* device is "READY", "RUNNING" and "COMMUNICATING" */
          /* Start cyclic demo with I/O Data-Transfer and packet data transfer */
          unsigned long ulCycCnt = 0;
          //uint32_t ulTriggerCount = 0;

          /* Cyclic I/O and packet handling for 'ulCycCnt'times */
          while( ulCycCnt < DEMO_CYCLES)
          {
            /* Start and trigger watchdog function, if necessary */
            //DEV_TriggerWatchdog(ptChannel, CIFX_WATCHDOG_START, &ulTriggerCount); 
            
            /* Handle I/O data transfer */
            IODemo      (ptChannel);

            /* Handle rcX packet transfer */
            #ifdef FIELDBUS_INDICATION_HANDLING
              Fieldbus_HandleIndications( ptChannel);  
            #else
              PacketDemo ( ptChannel);
            #endif

            ulCycCnt++;
          }
          
          /* Stop watchdog function, if it was previously started */
          //DEV_TriggerWatchdog(ptChannel, CIFX_WATCHDOG_STOP, &ulTriggerCount); 
        }

        /* Switch OFF the BUS communication / dont't wait */
        lRet = DEV_BusState( ptChannel, CIFX_BUS_STATE_OFF, &ulState, 0);

        /* Signal Host application is not available anymore / don't wait */
        lRet = DEV_SetHostState( ptChannel, CIFX_HOST_STATE_NOT_READY, 0);
      }
    
      /* Uninitialize interrupt */
      DEV_UninitializeInterrupt ( &tDevInstance);
    }
  }

  /* Cleanup all used nenory areas and pointers */
  cifXTkHWFunctions_UninitializeDataStructures( &tDevInstance);

  /* cifXTkHWFunctions cleanup */
  cifXTkHWFunctions_FreeDPMPointer( pbDPM);

	return lDemoRet;
}
